/**
 * \file keyman_caam.c
 *
 * \brief IMX6 specific implementation of keyman functions defined in keyman_arch.h
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <sdc_random.h>

#include <sdc/arch/errors.h>
#include <sdc/arch/common.h>
#include <sdc/arch/perm.h>

#include <keyman.h>
#include "keyman_caam.h"
#include <private/keyman_intern.h>
#include <private/keyman_arch.h>
#include <private/base64.h>

/** Gets length of string literal at build time */
#define SLEN(x) (sizeof((x)) - 1)
#define U32MAXDECSTRLEN (SLEN("4294967295"))


#define CAAM_PLAIN_KEY_AUTH_CONF_USER    (CAAM_KEY_FLAG_AUTH_MEDIUM|CAAM_KEY_FLAG_CONFIDENT_MEDIUM|CAAM_IMX6_USER_FLAGS__NORMAL_KEYSTORE_KEY)
#define CAAM_RANDOM_KEY_AUTH_CONF_USER    (CAAM_KEY_FLAG_AUTH_MEDIUM|CAAM_KEY_FLAG_CONFIDENT_MEDIUM|CAAM_IMX6_USER_FLAGS__NORMAL_KEYSTORE_KEY)
#define CAAM_INSEC_PRODUCT_KEY_AUTH_CONF_USER    (CAAM_KEY_FLAG_AUTH_MEDIUM|CAAM_KEY_FLAG_CONFIDENT_LOW|CAAM_IMX6_USER_FLAGS__PRODUCT_KEYSTORE_KEY)

static sdc_error_t error_from_ioctl_errno(int err);

static sdc_error_t keyman_caam_read_keystore_file(FILE *fp,
                                                  struct disc_key **dkeys, uint32_t *nkeys)
{
    sdc_error_t err = SDC_OK;
    uint32_t n_keys = 0;
    struct disc_key *d_keys = NULL;
    struct disc_key *temp;
    size_t line_max_len;
    char *line, *line_res; /* Stores a line of the key_store file */
    size_t read_len;
    int ext_len;

    *dkeys = NULL;
    *nkeys = 0;

    /* Buffer for lines of file:
     * Get max number of characters expected on a line and double it
     * [[[length of max val] + '\t'] * 3] +
     * [B64 data len] + '\t'
     * [B64 ext len] + '\n' */
    line_max_len = ((U32MAXDECSTRLEN + 1) * 3) +
                   BLOB_B64_LEN + 1 +
                   EXT_B64_LEN + 1;

    line = malloc(line_max_len);
    if (line == NULL) {
        return SDC_NO_MEM;
    }

    /* Start at the beginning of the file */
    rewind(fp);

    while (feof(fp) == 0) {
        /* Increase allocation for new key */
        temp = realloc(d_keys, (n_keys + 1) * sizeof(struct disc_key));
        if (temp == NULL) {
            err = SDC_NO_MEM;
            break;
        }
        d_keys = temp;

        /* Read data */
        line_res = fgets(line, line_max_len, fp);
        if (line_res == NULL) {
            continue;
        }

        d_keys[n_keys].kid = strtoul(line_res, &line_res, 10);
        if (*line_res++ != '\t' || !isdigit(*line_res)) {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }

        d_keys[n_keys].uid = strtoul(line_res, &line_res, 10);
        if (*line_res++ != '\t' || !isdigit(*line_res)) {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }

        d_keys[n_keys].gid = strtoul(line_res, &line_res, 10);
        if (*line_res++ != '\t') {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }

        /* Get length of base64 data */
        read_len = 0;
        while (!isspace(*(line_res + read_len)) && /* newline */
               !isblank(*(line_res + read_len)) && /* tab, space */
               *(line_res + read_len) != '\0') {
            read_len++;
        }

        if (read_len > BLOB_B64_LEN) {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }
        memcpy(d_keys[n_keys].blob_b64, line_res, read_len);
        d_keys[n_keys].blob_b64[read_len] = '\0';

        line_res += read_len;

        if (base64_decode_len(d_keys[n_keys].blob_b64) >
            (BLOB_LEN + 1)) {
            /* Base64 data too long */
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }

        /* Convert from disc encoding */
        d_keys[n_keys].blob_len =
            base64_decode(d_keys[n_keys].blob,
                          d_keys[n_keys].blob_b64);
        if ((d_keys[n_keys].blob_len < CAAM_KM_ENCAP_KEY_SIZE_MIN) ||
            (d_keys[n_keys].blob_len > CAAM_KM_ENCAP_KEY_SIZE_MAX)) {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }


        /* Get length of base64 extension data (optional) */
        read_len = 0;
        if (isblank(*(line_res))) { /* tab, space */
            line_res++;
            while (!isspace(*(line_res + read_len)) && /* newline */
                   !isblank(*(line_res + read_len)) && /* tab, space */
                   *(line_res + read_len) != '\0') {
                read_len++;
            }
        }

        d_keys[n_keys].with_ext = false;
        if (read_len > 0) {
            /* has extension data */
            d_keys[n_keys].with_ext = true;

            if (read_len > EXT_B64_LEN) {
                err = SDC_KEY_STORAGE_CORRUPT;
                break;
            }
            memcpy(d_keys[n_keys].ext_b64, line_res, read_len);
            d_keys[n_keys].ext_b64[read_len] = '\0';

            line_res += read_len;

            if (base64_decode_len(d_keys[n_keys].ext_b64) >
                (EXT_LEN + 1)) {
                /* Base64 data too long */
                err = SDC_KEY_STORAGE_CORRUPT;
                break;
            }

            /* Convert from disc encoding */
            ext_len = base64_decode(d_keys[n_keys].ext.as_bytes,
                                    d_keys[n_keys].ext_b64);
            if (ext_len != EXT_LEN) {
                err = SDC_KEY_STORAGE_CORRUPT;
                break;
            }

            /* currently modifier and additional data is not supported */
            if ((d_keys[n_keys].ext.as_struct.modifier_bytes != 0) ||
                (d_keys[n_keys].ext.as_struct.add_data_bytes != 0)) {
                err = SDC_KEY_STORAGE_CORRUPT;
                break;
            }
        }

        /* check for unexpected line end -- incomplete data */
        if (*(line_res) == '\0') {
            err = SDC_KEY_STORAGE_CORRUPT;
            break;
        }

        n_keys++;
    }

    free(line);

    if (err == SDC_OK) {
        if (n_keys == 0) {
            free(d_keys);
            d_keys = NULL;
        }

        *dkeys = d_keys;
        *nkeys = n_keys;
    } else {
        free(d_keys);
    }

    return err;
}

static sdc_error_t keyman_caam_write_keystore_file(FILE *fp,
                                                   const struct disc_key *dkeys, uint32_t nkeys)
{
    uint32_t i;

    for (i = 0; i < nkeys; i++) {
        if (dkeys[i].with_ext) {
            fprintf(fp, "%u\t%u\t%u\t%s\t%s\n",
                    dkeys[i].kid,
                    dkeys[i].uid,
                    dkeys[i].gid,
                    dkeys[i].blob_b64,
                    dkeys[i].ext_b64);
        } else {
            fprintf(fp, "%u\t%u\t%u\t%s\n",
                    dkeys[i].kid,
                    dkeys[i].uid,
                    dkeys[i].gid,
                    dkeys[i].blob_b64);
        }
    }

    fflush(fp);

    return SDC_OK;
}

static sdc_error_t keyman_caam_install_key(int keyman_dev_fd,
                                           struct caam_km_install_keystore_key *install_iodata)
{

    sdc_error_t err = SDC_OK;
    int res, e;

    /* Call the kernel's CAAM_KM_PREPARE_KEY */
    res = ioctl(keyman_dev_fd, CAAM_KM_INSTALL_KEYSTORE_KEY, install_iodata);
    e = errno;

    if (res != 0) {
        err = error_from_ioctl_errno(e);
        if (err == SDC_NOT_SUPPORTED) {
            /* key length is not supported */
            err = SDC_KEY_INVALID;
        }
    }

    return err;
}


static sdc_error_t keyman_caam_load_key(int keyman_dev_fd,
                                        const struct disc_key *dkey)
{
    struct caam_km_load_keystore_key data;
    uint32_t load_ext_flag;
    int res, err;

    memset(&data, 0, sizeof(struct caam_km_load_keystore_key));

    /* Set the load data to what we prepared */
    data.load_flags = CAAM_KM_LOAD_ACTIVATE;
    load_ext_flag = CAAM_KM_LOAD_NO_EXT_SPEC;
    data.kid = dkey->kid;
    data.spec.uid = dkey->uid;
    data.spec.gid = dkey->gid;
    data.key_encap_bytes = dkey->blob_len;
    data.key_encap = (uint8_t *)dkey->blob;

    if (dkey->with_ext) {
        load_ext_flag = CAAM_KM_LOAD_WITH_EXT_SPEC;
        data.spec.key_flags = dkey->ext.as_struct.key_flags;
        data.spec.key_bits = dkey->ext.as_struct.key_bits;
        memcpy(&data.spec.perms, &dkey->ext.as_struct.perms,
               sizeof(struct caam_key_perms));
    }

    data.load_flags |= load_ext_flag;

    /* Call the kernel's CAAM_KM_LOAD_KEY */
    res = ioctl(keyman_dev_fd, CAAM_KM_LOAD_KEYSTORE_KEY, &data);
    err = errno;

    if (res != 0) {
        return error_from_ioctl_errno(err);
    }

    return SDC_OK;
}

static sdc_error_t keyman_caam_remove_key(sdc_key_id_t kid, uid_t uid, gid_t gid)
{
    struct caam_km_remove_keystore_key data;
    int res, err;
    int fd;

    data.kid = kid;
    data.uid = uid;
    data.gid = gid;

    /* Open the keyman device */
    fd = open(KEYMAN_KEYMAN_DEVICE, O_RDWR, 0);
    if (fd < 0) {
        return SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
    }

    /* Call the kernel's CAAM_KM_REL_KEY */
    res = ioctl(fd, CAAM_KM_REL_KEYSTORE_KEY, &data);
    err = errno;

    close (fd);

    if (res != 0) {
        return error_from_ioctl_errno(err);
    }

    return SDC_OK;
}

static sdc_error_t keyman_caam_transfer_keys_to_kernel(
    const struct disc_key *dkeys, uint32_t nkeys,
    size_t *loaded_keys,
    size_t *existing_keys)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmperr;
    size_t i;
    int fd;

    /* Open the keyman device */
    fd = open(KEYMAN_KEYMAN_DEVICE, O_RDWR, 0);
    if (fd < 0) {
        return SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
    }

    for (i = 0; i < nkeys; i++) {
        tmperr = keyman_caam_load_key(fd, &dkeys[i]);
        if (tmperr == SDC_OK) {
            *loaded_keys += 1;
        }
        if (tmperr == SDC_KID_EXISTS) {
            *existing_keys += 1;
        }
        if ((tmperr != SDC_OK) && (tmperr != SDC_KID_EXISTS)) {
            err = tmperr;
        }
    }

    close (fd);

    return err;
}

static sdc_error_t error_from_ioctl_errno(int err)
{
    switch (err) {
    case ENOKEY:
        return SDC_KID_NOT_AVAILABLE;
    case EEXIST:
        return SDC_KID_EXISTS;
    case ENOSPC:
        return SDC_NO_MEM;
    case EACCES:
        return SDC_ACCESS_DENIED;
    case EKEYREVOKED:
        return SDC_KEY_TAMPERED;
    case EOPNOTSUPP:
        return SDC_OP_NOT_SUPPORTED;
    case EBUSY:
        return SDC_KEY_LOCKED;
    case EPERM:
        return SDC_PERM_INVALID;
    case EBADMSG:
        return SDC_KEY_INVALID;
    case EMSGSIZE:
        return SDC_KEYLEN_INVALID;
    case EBADF:
        return SDC_IMX6_INVALID_DEVICENODE;
    default:
        return SDC_IMX6_KEYMAN_DEVICENODE_IOCTL_FAILED;
    }
}

/* return SDC_OK if not found, SDC_KID_EXISTS if found */
static sdc_error_t keyman_caam_kid_not_in_dkeys(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t kid, size_t *idx)
{
    size_t i = 0;
    sdc_error_t found = SDC_OK;

    while ((i<ctx->disc_keys_num) && (found == SDC_OK)) {
        if (ctx->disc_keys[i].kid == kid) {
            found = SDC_KID_EXISTS;
            if (idx)
                *idx = i;
        }
        i++;
    }

    return found;
}

static sdc_error_t keyman_caam_fill_install_iodata(
    struct caam_km_install_keystore_key *install_iodata,
    sdc_key_id_t key_id_requested,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    const libkeyman_key_spec_t *key_spec,
    bool activate_immediately,
    bool import,
    const sdc_keysecret_t *key_secret,
    uint32_t auth_conf_user_flags
    )
{
    sdc_error_t err = SDC_OK;
    uid_t uid = (uid_t)-1;
    gid_t gid = (gid_t)-1;
    size_t key_bits = 0;
    size_t key_bytes = 0;

    install_iodata->install_flags = 0;

    if ((key_spec->fmt != SDC_KEY_FMT_SIMPLE_SYM) &&
        (key_spec->fmt != SDC_KEY_FMT_RSA_PRIVATE) &&
        (key_spec->fmt != SDC_KEY_FMT_RSA_PUBLIC))
        err = SDC_NOT_SUPPORTED;

    if (key_secret && key_secret->secret) {
        if ((key_spec->fmt == SDC_KEY_FMT_SIMPLE_SYM) &&
            (key_secret->enc != SDC_KEY_ENC_PLAIN))
            return SDC_KEY_ENC_INVALID;

        if (((key_spec->fmt == SDC_KEY_FMT_RSA_PRIVATE) ||
             (key_spec->fmt == SDC_KEY_FMT_RSA_PUBLIC)) &&
            (key_secret->enc != SDC_KEY_ENC_DER) &&
            (key_secret->enc != SDC_KEY_ENC_PEM))
            return SDC_KEY_ENC_INVALID;
    } else {
        if ((key_spec->fmt == SDC_KEY_FMT_RSA_PRIVATE) ||
            (key_spec->fmt == SDC_KEY_FMT_RSA_PUBLIC)) {
            /* random not supported for RSA */
            err = SDC_NOT_SUPPORTED;
        } else {
            /* random keys can't be imported */
            if (import)
                err = SDC_INTERNAL_ERROR;
        }
    }

    if (err == SDC_OK)
        err = sdc_key_len_get_bytes(key_spec->len, &key_bytes);

    if (err == SDC_OK)
        err = sdc_key_len_get_bits(key_spec->len, &key_bits);

    if ((err == SDC_OK) &&
        ((key_bytes > CAAM_KM_KEYSECRET_LEN_MAX) ||
         (key_bytes < CAAM_KM_KEYSECRET_LEN_MIN)))
        err = SDC_KEYLEN_INVALID;

    if (activate_immediately)
        install_iodata->install_flags |= CAAM_KM_INSTALL_ACTIVATE;
    else
        install_iodata->install_flags |= CAAM_KM_INSTALL_DONT_ACTIVATE;

    /* Set requested key id */
    install_iodata->kid = key_id_requested;

    /* set range to look for key id */
    install_iodata->kid_min = key_id_first;
    install_iodata->kid_max = key_id_last;

    install_iodata->spec.key_bits = key_bits;

    install_iodata->spec.key_flags = CAAM_KEY_FLAG_TYPE_KEYSTORE |
                                     auth_conf_user_flags;

    if (err == SDC_OK) {
        switch (key_spec->fmt) {
        case SDC_KEY_FMT_SIMPLE_SYM:
            install_iodata->spec.key_flags |= CAAM_KEY_FLAG_STOR_BLACK |
                                              CAAM_KEY_FLAG_SECRET_BYTES_PLAIN;

            /*
             * when installing a key from key_data, the key_data_len needs to be equal
             * to length defined by key key_spec->len
             * We need to skip this plausibility check in case of random or imported key
             */
            if (key_secret && key_secret->secret && (!import) &&
                    (key_bytes != key_secret->secret_len))
                err = SDC_INVALID_PARAMETER;
            break;
        case SDC_KEY_FMT_RSA_PRIVATE:
            /* public and private keys need to be red to be accessible in kernel */
            install_iodata->spec.key_flags |= CAAM_KEY_FLAG_STOR_RED |
                                              CAAM_KEY_FLAG_SECRET_RSA_BER_PRI;
            break;
        case SDC_KEY_FMT_RSA_PUBLIC:
            /* public and private keys need to be red to be accessible in kernel */
            install_iodata->spec.key_flags |= CAAM_KEY_FLAG_STOR_RED |
                                              CAAM_KEY_FLAG_SECRET_RSA_BER_PUB;
            break;
        default:
            err = SDC_INTERNAL_ERROR;
        }
    }

    if (key_secret && key_secret->enc == SDC_KEY_ENC_PEM)
        install_iodata->install_flags |= CAAM_KM_INSTALL_PEM;

    if (err == SDC_OK)
        err = sdc_permissions_get_gid(key_spec->permissions, &gid);

    if (err == SDC_OK)
        err = sdc_permissions_get_uid(key_spec->permissions, &uid);

    if (err == SDC_OK) {
        /* Set the UID and GID */
        install_iodata->spec.uid = sdc_uid_to_caam_uid(uid);
        install_iodata->spec.gid = sdc_gid_to_caam_gid(gid);

        err = sdc_permissions_to_caam_key_perms(key_spec->permissions, &(install_iodata->spec.perms));
    }

    /* currently we don't use modifiers */
    install_iodata->spec.modifier = NULL;
    install_iodata->spec.modifier_bytes = 0;

    /* currently we don't use additional data */
    install_iodata->additional_key_data = NULL;
    install_iodata->additional_key_data_bytes = 0;

    /* Set input to key secret */
    install_iodata->key_secret_bytes = 0;
    install_iodata->key_secret = NULL;
    if (key_secret && key_secret->secret) {
        install_iodata->key_secret = key_secret->secret;
        install_iodata->key_secret_bytes = key_secret->secret_len;
        if (import)
            install_iodata->spec.key_flags |= CAAM_KEY_FLAG_SUBTYPE_IMPORTED;
        else
            install_iodata->spec.key_flags |= CAAM_KEY_FLAG_SUBTYPE_PLAIN;
    } else {
        install_iodata->spec.key_flags |= CAAM_KEY_FLAG_SUBTYPE_RANDOM;
    }

    return err;
}

static sdc_error_t keyman_caam_generate_storage_key_common(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id_range,     /* key_id_range->result==NULL in case of explicit kid */
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    bool import,
    const sdc_keysecret_t *key_secret,
    uint32_t auth_conf_user_flags)
{
    sdc_error_t err = SDC_OK;
    sdc_key_id_t init_kid = SDC_FLAG_INVALID_KID;
    struct caam_km_install_keystore_key install_iodata;
    int fd;
    struct disc_key *temp;
    struct disc_key new_key;
    size_t disc_keys_idx_new;
    size_t disc_keys_num_new;

    memset(&new_key, 0, sizeof(struct disc_key));
    new_key.blob_len = BLOB_LEN;

    err = keyman_intern_search_kid_init(key_id_range->first,
                                        key_id_range->last, &init_kid);

    if (err == SDC_OK) {
        err = keyman_caam_fill_install_iodata(
            &install_iodata,
            init_kid, key_id_range->first, key_id_range->last,
            key_spec,
            activate_immediately,
            import,
            key_secret,
            auth_conf_user_flags);
    }

    install_iodata.key_encap = NULL;
    install_iodata.key_encap_bytes = 0;

    if (persistent) {
        install_iodata.key_encap = (uint8_t *)&new_key.blob;
        install_iodata.key_encap_bytes = new_key.blob_len;
    }

    if (err == SDC_OK) {
        /* Open the keyman device */
        fd = open(KEYMAN_KEYMAN_DEVICE, O_RDWR, 0);
        if (fd < 0) {
            err = SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
        }

        /* search an available key id */
        if (err == SDC_OK) {
            err = keyman_caam_install_key(
                fd,
                &install_iodata);
        }

        if (err == SDC_OK) {
            /* copy found kid */
            new_key.kid = install_iodata.kid;
            new_key.uid = install_iodata.spec.uid;
            new_key.gid = install_iodata.spec.gid;
            new_key.blob_len = install_iodata.key_encap_bytes;

            /* copy ext data */
            new_key.with_ext = true;
            new_key.ext.as_struct.key_flags = install_iodata.spec.key_flags;
            new_key.ext.as_struct.key_bits = install_iodata.spec.key_bits;
            new_key.ext.as_struct.modifier_bytes = 0;
            new_key.ext.as_struct.add_data_bytes = 0;
            memcpy(&new_key.ext.as_struct.perms, &install_iodata.spec.perms,
                   sizeof(struct caam_key_perms));

            if (base64_encode_len(new_key.blob_len) > (BLOB_B64_LEN + 1)) {
                err = SDC_KEY_INVALID;
            }
        }

        if (err == SDC_OK) {
            /* encode blob and ext data */
            base64_encode(new_key.blob_b64,
                          new_key.blob,
                          new_key.blob_len);

            base64_encode(new_key.ext_b64,
                          new_key.ext.as_bytes,
                          EXT_LEN);
        }

        close(fd);
    }

    if ((err == SDC_OK) && (persistent)) {
        disc_keys_idx_new = ctx->disc_keys_num;
        disc_keys_num_new = ctx->disc_keys_num + 1;

        /* increase disc_keys memory */
        temp = realloc(ctx->disc_keys,
                       (disc_keys_num_new) * sizeof(struct disc_key));
        if (!temp) {
            err = SDC_NO_MEM;
        } else {
            ctx->disc_keys = temp;
        }

        /* copy new key to disc_keys */
        if (err == SDC_OK) {
            memcpy(&ctx->disc_keys[disc_keys_idx_new],
                   &new_key, sizeof(new_key));
            ctx->disc_keys_num = disc_keys_num_new;
        } else {
            /* the key could not be stored in disc_keys - remove it */
            (void)keyman_caam_remove_key(new_key.kid, new_key.uid, new_key.gid);
        }

    }

    if ((err == SDC_OK) && (key_id_range->result)) {
        *(key_id_range->result) = new_key.kid;
    }

    return err;
}


/* implementation of libkeyman internal arch dependent functions */
sdc_error_t keyman_arch_kernel_version_verify(char **version_info_str)
{
#ifdef CAAM_KM_GET_KERNEL_VERSION
    int fd;
    int res;
    sdc_error_t err = SDC_OK;
    struct caam_km_version kernel_version_info;
    uint32_t sw_feature_version;
    uint32_t sw_feature_subversion;

    if (version_info_str != NULL) {
        *version_info_str = NULL;
    }

    fd = open(KEYMAN_KEYMAN_DEVICE, O_RDONLY, 0);

    if (fd < 0) {
        fprintf(stderr,"Error: Couldn't open device\n");
        return SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
    }

    memset(&kernel_version_info, 0, sizeof(kernel_version_info));

    res = ioctl(fd, CAAM_KM_GET_KERNEL_VERSION, &kernel_version_info);
    if (res != 0) {
        /* kernel does not provide version info - initialize to 0 */
        kernel_version_info.version = 0;
        kernel_version_info.subversion = 0;
        kernel_version_info.ioctl_comp_version = 0;
    }

    /* version required by library */
    sw_feature_version = LIBKEYMAN_FEATURE_VERSION;
    sw_feature_subversion = LIBKEYMAN_FEATURE_SUBVERSION;

    /* generate info string */
    if (version_info_str != NULL) {
        *version_info_str = malloc(VERSION_INFO_STR_MAX);

        if (*version_info_str) {
            snprintf(*version_info_str, VERSION_INFO_STR_MAX,
                     "libKeyman kernel %d-%d ioctl-version %d, library %d-%d",
                     kernel_version_info.version,
                     kernel_version_info.subversion,
                     kernel_version_info.ioctl_comp_version,
                     sw_feature_version, sw_feature_subversion);
        }
    }

    /* check version info
     * the kernel has to provide at least the amount of features
     * as available when building/commiting the library
     *     version > sw_feature_version
     *     version == sw_feature_version && subversion >= sw_feature_subversion
     *
     * The ioctls need to be compatible to ioctls
     * available when building the library
     *  ioctl_comp_version <= sw_feature_version
     */
    if ((kernel_version_info.version < sw_feature_version) ||
        ((kernel_version_info.version == sw_feature_version) &&
         (kernel_version_info.subversion < sw_feature_subversion)) ||
        (kernel_version_info.ioctl_comp_version > sw_feature_version)) {
        err = SDC_INVALID_VERSION;
    }

    close(fd);
    return err;
#else
    return SDC_NOT_SUPPORTED;
#endif
}

sdc_error_t keyman_arch_ctx_alloc(libkeyman_ctx_t **ctx)
{
    *ctx = malloc(sizeof(libkeyman_ctx_t));
    if (!ctx)
        return SDC_NO_MEM;

    (*ctx)->disc_keys = NULL;
    (*ctx)->disc_keys_num = 0;

    return SDC_OK;
}

sdc_error_t keyman_arch_ctx_free(libkeyman_ctx_t *ctx)
{
    free(ctx->disc_keys);

    ctx->disc_keys = NULL;
    ctx->disc_keys_num = 0;

    free(ctx);

    return SDC_OK;
}

sdc_error_t keyman_arch_load_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                                 size_t *loaded_keys, size_t *existing_keys)
{
    FILE *fp;
    sdc_error_t err;
    struct disc_key *temp_disc_keys;
    size_t temp_disc_keys_num;

    *loaded_keys = 0;
    *existing_keys = 0;

    fp = fdopen(fd, "r");
    if (fp == NULL) {
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    err = keyman_caam_read_keystore_file(
        fp,
        &temp_disc_keys,
        &temp_disc_keys_num);

    if (err == SDC_OK) {
        free(ctx->disc_keys);

        ctx->disc_keys = temp_disc_keys;
        ctx->disc_keys_num = temp_disc_keys_num;
    }

    if (err == SDC_OK) {
        err = keyman_caam_transfer_keys_to_kernel(
            ctx->disc_keys,
            ctx->disc_keys_num,
            loaded_keys, existing_keys);
    }

    fclose(fp);

    return err;
}

sdc_error_t keyman_arch_store_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                                  size_t *stored_keys)
{
    FILE *fp;
    sdc_error_t err;

    fp = fdopen(fd, "w");
    if (fp == NULL) {
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    err = keyman_caam_write_keystore_file(fp, ctx->disc_keys,
                                          ctx->disc_keys_num);

    if (fsync(fileno(fp))) {
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    if (err == SDC_OK) {
        *stored_keys = ctx->disc_keys_num;
    }

    fclose(fp);

    return err;
}

sdc_error_t keyman_arch_empty_persistent_keystore(libkeyman_ctx_t *ctx, int fd)
{
    FILE *fp;
    sdc_error_t err;

    (void)ctx;

    fp = fdopen(fd, "w");
    if (fp == NULL) {
        return SDC_KEY_STORAGE_ACCESS_FAILED;
    }

    err = keyman_caam_write_keystore_file(fp, NULL, 0);

    fclose(fp);

    return err;
}

sdc_error_t keyman_arch_flush_keystore(libkeyman_ctx_t *ctx,
                                       libkeyman_keystore_cont_t keystore_content, bool force)
{
    (void)ctx;
    (void)keystore_content;
    (void)force;

    return SDC_NOT_SUPPORTED;
}

sdc_error_t keyman_arch_lock_unlock_keystore(libkeyman_ctx_t *ctx,
                                             libkeyman_keystore_cont_t keystore_content, bool lock)
{
    sdc_error_t sdc_err = SDC_OK;
    int fd;
    int err, res;

    (void)ctx;

    if ((keystore_content != LIBKEYMAN_KEYSTORE_ALL) ||
        (lock != false)) {
        return SDC_NOT_SUPPORTED;
    }

    /* Open the keyman device */
    fd = open(KEYMAN_KEYMAN_DEVICE, O_RDWR, 0);
    if (fd < 0) {
        return SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
    }


    res = ioctl(fd, CAAM_KM_UNLOCK_KEYSTORE, NULL);
    err = errno;

    if (res != 0) {
        sdc_err = error_from_ioctl_errno(err);
    }

    close (fd);

    return sdc_err;



}

sdc_error_t keyman_arch_generate_storage_key(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id_range,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret)
{
    uint32_t auth_conf_user_flags = (key_secret == NULL ?
                                     CAAM_RANDOM_KEY_AUTH_CONF_USER :
                                     CAAM_PLAIN_KEY_AUTH_CONF_USER);

    return keyman_caam_generate_storage_key_common(
               ctx,
               key_id_range,
               key_spec,
               persistent,
               activate_immediately,
               false,
               key_secret,
               auth_conf_user_flags);
}

static sdc_error_t get_import_auth_conf_user_flags(
    const sdc_keysecret_t *key_secret,
    uint32_t *auth_conf_user_flags)
{
    sdc_error_t err = SDC_OK;
    const struct caam_key_encap_info *encap_info;
    uint32_t authentitcity;
    uint32_t confidentiality;

    *auth_conf_user_flags = CAAM_IMX6_USER_FLAGS__NORMAL_KEYSTORE_KEY;

    if (!key_secret || key_secret->secret_len < CAAM_KEY_ENCAP_INFO_LEN) {
        err = SDC_IMX6_ENCAP_INVALID;
    } else {
        /* cast to void* to make LINT happy */
        const void *key_secret_secret = key_secret->secret;

        /* encap info is at the very start of key_data */
        encap_info = (const struct caam_key_encap_info *)key_secret_secret;

        /* use the same authenticity for imported keys */
        authentitcity = encap_info->encap_auth;
        *auth_conf_user_flags |= authentitcity;

        /* use the same confidentiality (but max medium) for imported keys */
        confidentiality = encap_info->encap_conf;
        if (confidentiality > CAAM_KEY_FLAG_CONFIDENT_MEDIUM)
            confidentiality = CAAM_KEY_FLAG_CONFIDENT_MEDIUM;
        *auth_conf_user_flags |= confidentiality;
    }

    return err;
}

sdc_error_t keyman_arch_import_storage_key(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id_range,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret)
{
    sdc_error_t err;
    uint32_t auth_conf_user_flags;

    err = get_import_auth_conf_user_flags(key_secret, &auth_conf_user_flags);
    if (err != SDC_OK)
        return err;

    return keyman_caam_generate_storage_key_common(
               ctx,
               key_id_range,
               key_spec,
               persistent,
               activate_immediately,
               true,
               key_secret,
               auth_conf_user_flags);
}

sdc_error_t keyman_arch_import_insecure_product_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret)
{
    sdc_key_id_range_t key_id_range = {
            .first = key_id,
            .last = key_id,
            .result = NULL,
    };

    /* always persistent*/
    return keyman_caam_generate_storage_key_common(
               ctx,
               &key_id_range,
               key_spec,
               true,
               activate_immediately,
               false,
               key_secret,
               CAAM_INSEC_PRODUCT_KEY_AUTH_CONF_USER);
}

sdc_error_t keyman_arch_activate_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id)
{
    sdc_error_t sdc_err = SDC_OK;
    struct caam_km_activate_keystore_key data;
    int fd;
    int err, res;

    (void)ctx;

    /* Open the keyman device */
    fd = open(KEYMAN_KEYMAN_DEVICE, O_RDWR, 0);
    if (fd < 0) {
        return SDC_IMX6_OPEN_KEYMAN_DEVICENODE_FAILED;
    }

    data.kid = key_id;

    res = ioctl(fd, CAAM_KM_ACTIVATE_KEYSTORE_KEY, &data);
    err = errno;

    if (res != 0) {
        sdc_err = error_from_ioctl_errno(err);
    }

    close (fd);

    return sdc_err;
}

sdc_error_t keyman_arch_remove_storage_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    uid_t uid, gid_t gid,
    bool *result_is_persistent
    )
{
    sdc_error_t err;
    sdc_error_t tmperr;
    size_t pers_idx;
    size_t shift_count;

    struct disc_key *temp = NULL;

    err = keyman_caam_remove_key(key_id, uid, gid);

    if (err == SDC_OK) {
        tmperr = keyman_caam_kid_not_in_dkeys(ctx, key_id, &pers_idx);
        if (tmperr != SDC_KID_EXISTS) {
            *result_is_persistent = false;
        } else {
            *result_is_persistent = true;

            /* we need to shift keys after the removed key */
            shift_count = (ctx->disc_keys_num - 1) - pers_idx;
            while (shift_count > 0) {
                memcpy(&ctx->disc_keys[pers_idx],
                       &ctx->disc_keys[pers_idx+1],
                       sizeof(struct disc_key));
                pers_idx++;
                shift_count--;
            }
            ctx->disc_keys_num = ctx->disc_keys_num - 1;

            if (ctx->disc_keys_num == 0) {
                free(ctx->disc_keys);
                ctx->disc_keys = NULL;
            } else {
                temp = realloc(ctx->disc_keys,
                               ctx->disc_keys_num * sizeof(struct disc_key));
                if (temp) {
                    ctx->disc_keys = temp;
                } /* else - something went wrong - ignore it */
            }
        }
    }

    return err;
}

